##    Copyright (C) 2006 Sepp Hochreiter (hochreit@cs.tu-berlin.de),
##                       Djork-Arne Clevert (clevert@cs.tu-berin.de),
##                       Klaus Obermayer (oby@cs.tu-berlin.de)
##    Berlin University of Technology,
##    Institute for Software Engineering and Theoretical Computer Science 
##    The software is maintained by Djork-Arné Clevert. 
##    We offer a first implementation of the new 
##    ``Factor Analysis for Robust Microarray Summarization'' (FARMS) algorithm.
##    This program is free software; you can redistribute it and/or modify it under 
##    the terms of the GNU General Public License as published by the Free Software 
##    Foundation; either version 2 of the License, or (at your option) any later version. 
##    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
##    without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
##    See the GNU General Public License for more details.
##    If you use this library, please cite:
##
##    @article{SeppHochreiter02102006,
##		author = {Hochreiter, Sepp and Clevert, Djork-Arne and Obermayer, Klaus},
##		title = {{A new summarization method for Affymetrix probe level data}},
##		journal = {Bioinformatics},
##		volume = {},
##		number = {},
##		pages = {btl033},
##		doi = {10.1093/bioinformatics/btl033},
##		year = {2006},
##		URL = {http://bioinformatics.oxfordjournals.org/cgi/content/abstract/btl033v1},
##		eprint = {http://bioinformatics.oxfordjournals.org/cgi/reprint/btl033v1.pdf}
##		}




## loads affy library
require(affy)




generateExprSet.methods <<- c(generateExprSet.methods,'farms') 
express.summary.stat.methods <<- c(express.summary.stat.methods,'farms')






exp.farms<-function(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "quantiles", weight, mu, scale, cyc, tol, ...)
{
if (missing(weight)){weight<-8}
if (missing(mu)){mu<-0}
if (missing(scale)){scale<-1.5}
if (missing(cyc)){cyc<-100}
if (missing(tol)){tol<-0.00001}

    res <- expresso(object, bgcorrect.method=bgcorrect.method, pmcorrect.method=pmcorrect.method, 
        normalize.method=normalize.method, summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu, scale=scale, cyc=cyc, tol=tol))
    return(res)
} 


q.farms<-function (object, weight, mu, scale, cyc, tol, ...) 
{
if (missing(weight)){weight<-8}
if (missing(mu)){mu<-0}
if (missing(scale)){scale<-2}
if (missing(cyc)){cyc<-100}
if (missing(tol)){tol<-0.00001}

    res <- expresso(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "quantiles", summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu, scale=scale, cyc=cyc, tol=tol))
    return(res)
}

l.farms<-function (object, weight, mu, scale, cyc, tol, ...) 
{
if (missing(weight)){weight<-8}
if (missing(mu)){mu<-0}
if (missing(scale)){scale<-1.5}
if (missing(cyc)){cyc<-100}
if (missing(tol)){tol<-0.00001}

    res <- expresso(object, bgcorrect.method = "none", pmcorrect.method = "pmonly", 
        normalize.method = "loess", summary.method = "farms", 
        summary.param=list(weight=weight, mu=mu, scale=scale, cyc=cyc, tol=tol))
    return(res)
}

generateExprVal.method.farms<- function(probes,weight,mu,scale,cyc,tol,...){
if (missing(weight)){weight<-8}
if (missing(mu)){mu<-0}
if (missing(scale)){scale<-1.5}
if (missing(cyc)){cyc<-100}
if (missing(tol)){tol<-0.00001}
## probes - data matrix
## weight - hyperparameter default (8)
## mu - hyperparameter default (0)
## scale - scaling parameter for quantiles- (1.5) and 
## loess-normalization (2)
  
## tol - termination tolerance (default = 0.00001)
  
## cyc - maximum ber og cycles of EM (default 100)
## L - factor loadings
## Ph - diagonal uniqueness matrix
L_old<-0
M <-  ncol(probes)
## log2 probes
x<-log2(t(probes))
n <- length(x[,1])
y_v<-colMeans(x)
xmean<- cbind(seq(1, 1, length=n))%*%y_v
X<-x-xmean  ## center data (0 mean)
XX<-t(X)%*%X/n
XX<-(XX+t(XX))/2 ## XX is now positive definit
cX<-cov(X)
diagXX<-diag(XX)
diagcX<-diag(cX)
eigen_v<-eigen(cX, symmetric = TRUE) ## calculate eigenvectors 
L<-abs(eigen_v$values[1]*eigen_v$vectors[,1]) ## L init 
LL<-L*L
Ph<-diagcX-LL
alpha<-weight/mean(diagcX)
bbeta<-mu*alpha
if (min(Ph)<0){Ph<-0.1*min(diagcX)+diagcX-(diagcX[which.min(Ph)]/LL[which.min(Ph)])*LL} ## avoids negative Ph 
for (i in 1:cyc){
# E Step
PsiL<-1/Ph*L
a<-as.vector(1+t(L)%*%PsiL)
bar<-PsiL/a
beta<-t(bar)
XXbeta<-XX%*%bar
EZZ<-1-beta%*%L+beta%*%XXbeta
t_XXbeta<-XXbeta+Ph*bbeta
t_EZZ<-as.vector(EZZ)+Ph*alpha
## M Step
L<-t_XXbeta/t_EZZ
neg<-which(L<0, arr.ind = TRUE) 
L[neg]<-0    ## rectified gaussian prior
Ph<-diagXX-XXbeta*L+Ph*alpha*L*(bbeta-L) 
if (abs(sqrt(sum(L^2))-sqrt(sum(L_old^2)))< tol){break}
L_old<-L
}
c<-X%*%bar ## hidden variable c - factor
Lambda<-mean(L) ## mean factor loading
Mean<-mean(y_v) ## mean 
express<-t(c)*Lambda*scale+Mean
SNR<-1/a
return(list(exprs=as.numeric(express),se.exprs=as.numeric(SNR)))
}





# I/NI calls

makeINICalls<-function(data.farms){
SNR<-se.exprs(data.farms[,1])
return(sum(SNR>0.5)/length(SNR))				# number of informative genes
}


showINIPlot<-function(data.farms){
# plot var(z|x) of FARMS summarized probe sets
SNR<-se.exprs(data.farms[,1])
truehist(SNR,col="lightgrey",border="darkgrey",xlab="Var(z|x)",ylab="density")
abline(v = .5, lty = 1,col="black")
}


reduceDataset<-function(data.farms){
SNR<-se.exprs(data.farms[,1])
IN.data <- data.farms[SNR<.5,]
return(IN.data) # return reduced data set				
}

















